/***************************************************************************
*
* Copyright 2012 Valeo
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
****************************************************************************/

#include <cassert>

#include "Log.h"
#include "config.h"
#include "IScene.h"
#include "InputManager.h"
#include "GraphicSystems/BaseGraphicSystem.h"
#include "Configuration.h"


InputManager::InputManager(IScene* s, Configuration &config) :
    m_pGraphicSystem(NULL),
    m_config(config),
    m_pScene(s),
    m_KbdFocus(NULL),
    m_PointerFocus(NULL),
    m_TouchFocus(NULL),
    m_pKeyboardCount(0),
    m_pPointerCount(0),
    m_pTouchCount(0)
{
    assert(s != NULL);
    strcpy(m_seatName, DEFAULT_SEAT_NAME);
    pthread_mutex_init(&m_mutex, NULL);
}

InputManager::~InputManager()
{
    m_pScene = NULL;
    pthread_mutex_destroy(&m_mutex);
}


/**
 * @brief Set the keyboard focus on a specific surface.
 */
bool InputManager::setKeyboardFocusOn(unsigned int surfId)
{
    bool ret;
    Surface * surf;

    surf = m_pScene->getSurface(surfId);
    if (surf == NULL)
    {
        LOG_WARNING("InputManager",
                    "Called with unknown surface, ID=" << surfId);
        ret = false;
    }
    else
    {
        if (surf->isInputEventAcceptedFrom(INPUT_DEVICE_KEYBOARD))
        {
            _setKbdFocus(surf);
            ret = true;
        }
        else
        {
            // Hum, the surface does not accept kbd events but we've been asked to set the focus on it
            _setKbdFocus(NULL);
            ret = false;
            LOG_WARNING("InputManager",
                        "Called with surface, ID=" << surfId << ". "
                        "This surface does not accept keyboard event !");
        }
    }

    return ret;
}

/**
 * @brief Get the identifier of the surface which has the keyboard focus
 */
unsigned int InputManager::getKeyboardFocusSurfaceId()
{
    Surface *surf;
    unsigned int surfId;

    surf = _getKbdFocus();
    if (!surf)
    {
        surfId = GraphicalObject::INVALID_ID;
    }
    else
    {
        surfId = surf->getID();
    }

    return surfId;
}

/**
 * @brief Set the pointer focus on a specific surface.
 */
bool InputManager::setInputFocusOn(unsigned int surfId, ilmInputDevice device_type, bool is_set)
{
    bool ret = true;
    if (device_type & ILM_INPUT_DEVICE_KEYBOARD)
    {
        if (true == is_set)
        {
            ret = setKeyboardFocusOn(surfId);
        }
        else
        {
            _setKbdFocus(NULL);
        }
    }
    if (device_type & ILM_INPUT_DEVICE_POINTER)
    {
        if (true == is_set)
        {
            ret = setPointerFocusOn(surfId);
        }
        else
        {
            _setPointerFocus(NULL);
        }
    }

    if (device_type & ILM_INPUT_DEVICE_TOUCH)
    {
        if (true == is_set)
        {
            Surface * surf;

            surf = m_pScene->getSurface(surfId);
            if (surf == NULL)
            {
                LOG_WARNING("InputManager",
                            "Called with unknown surface, ID=" << surfId);
                ret = false;
            }
            _setTouchFocus(surf);
        }
        else
        {
            _setTouchFocus(NULL);
        }
    }
    return ret;
}

bool InputManager::setPointerFocusOn(unsigned int surfId)
{
    bool ret;
    Surface * surf;

    surf = m_pScene->getSurface(surfId);
    if (surf == NULL)
    {
        LOG_WARNING("InputManager", 
                    "Called with unknown surface, ID=" << surfId);
        ret = false;
    }
    else
    {
        if (surf->isInputEventAcceptedFrom(INPUT_DEVICE_POINTER))
        {
            _setPointerFocus(surf);
            ret = true;
        }
        else
        {
            // Hum, the surface does not accept pointer events but we've been asked to set the focus on it
            _setPointerFocus(NULL);
            ret = false;
            LOG_WARNING("InputManager",
                        "Called with surface ID=" << surfId << ". "
                        "This surface does not accept pointer event !");
        }
    }


    return ret;
}

/**
 * @brief Get the identifier of the surface which has the pointer focus
 */
unsigned int InputManager::getPointerFocusSurfaceId()
{
    Surface *surf;
    unsigned int surfId;

    surf = _getPointerFocus();
    if (!surf)
    {
        surfId = GraphicalObject::INVALID_ID;
    }
    else
    {
        surfId = surf->getID();
    }

    return surfId;
}

bool InputManager::updateInputEventAcceptanceOn(unsigned int surfId, InputDevice devices, bool accept)
{
    bool ret;
    Surface * surf;

    surf = m_pScene->getSurface(surfId);
    if (surf == NULL)
    {
        LOG_WARNING("InputManager",
                    "Called with unknown surface, ID=" << surfId);
        ret = false;
    }
    else
    {
        surf->updateInputEventAcceptanceFrom(devices, accept);
        ret = true;
    }

    return ret;
}



/**
 * @brief Report keyboard event.
 * @param[in] state The state of the key. Can be either INPUT_STATE_PRESSED or INPUT_STATE_RELEASED
 * @param[in] keyId A uniq identifier for the key being reported.
 * @return The Surface to which to report the event, or NULL
 */
Surface * InputManager::reportKeyboardEvent(InputEventState state, long keyId)
{
    Surface * elected;

    switch (state)
    {
    case INPUT_STATE_PRESSED:
        elected = _getKbdFocus();
        m_KeyMap[keyId] = elected;
        break;

    case INPUT_STATE_RELEASED:
        elected = m_KeyMap[keyId];
        break;

    default:
        elected = NULL;
        LOG_WARNING("InputManager",
                    "Invalid input state, value=" << state);
        break;
    }
    if (NULL != elected)
    {
        if (false == elected->checkSeatAcceptance(m_seatName))
        {
            std::string seatName(m_seatName);
            LOG_WARNING("InputManager", "Ignoring keyboard event because Surface:"
                   << elected->getID() << " does not accept inputs from seat:" << seatName);
            elected = NULL;
        }
    }
    return elected;
}


/**
 * @return The Surface to which to report the event, or NULL
 */
Surface * InputManager::reportPointerEvent(Point& p)
{
    Surface * elected;
    if (p.x == -1 && p.y == -1)
    {
        elected = _getPointerFocus();
        LOG_DEBUG("InputManager",
                  "Do not check for pointer focus, because coordinate -1,-1");
        if (NULL != elected)
        {
            if (false == elected->checkSeatAcceptance(m_seatName))
            {
                std::string seatName(m_seatName);
                LOG_WARNING("InputManager", "Ignoring pointer event because Surface:"
                       << elected->getID() << " does not accept inputs from seat:" << seatName);
                elected = NULL;
            }
        }
    }
    else
    {
        switch (p.state)
        {
        case INPUT_STATE_PRESSED:
            elected = electSurfaceForPointerEvent(p.x, p.y,
                                                  INPUT_DEVICE_POINTER);
            _setPointerFocus(elected);
            break;

        case INPUT_STATE_OTHER:
        case INPUT_STATE_MOTION:
        case INPUT_STATE_RELEASED:
            elected = _getPointerFocus();
            if (elected != NULL)
            {
                if (!transformGlobalToLocalCoordinates(elected, p.x, p.y))
                    elected = NULL;
            }
            break;

        case INPUT_STATE_AXIS:
            elected = _getPointerFocus();
            break;

        default:
            elected = NULL;
            LOG_WARNING("InputManager",
                        "Invalid input state, value=" << p.state);
        }
    }
    return elected;
}

Surface * InputManager::reportPointerEvent(Point& p, int displayID)
{
    Surface * elected;
    if (p.x == -1 && p.y == -1)
    {
        elected = _getPointerFocus();
        LOG_DEBUG("InputManager",
                  "Do not check for pointer focus, because coordinate -1,-1");
        if (NULL != elected)
        {
            if (false == elected->checkSeatAcceptance(m_seatName))
            {
                std::string seatName(m_seatName);
                LOG_WARNING("InputManager", "Ignoring pointer event because Surface:"
                       << elected->getID() << " does not accept inputs from seat:" << seatName);
                elected = NULL;
            }
        }
    }
    else
    {
        switch (p.state)
        {
        case INPUT_STATE_PRESSED:
            if (INVALID_DISPLAY_ID == displayID)
            {
                elected = electSurfaceForPointerEvent(p.x, p.y,
                                                INPUT_DEVICE_POINTER);
            }
            else
            {
                elected = electSurfaceForPointerEventOnDisplay(p.x, p.y,
                                                  INPUT_DEVICE_POINTER, displayID);
            }
            _setPointerFocus(elected);
            break;

        case INPUT_STATE_OTHER:
        case INPUT_STATE_MOTION:
        case INPUT_STATE_RELEASED:
            elected = _getPointerFocus();
#ifdef WITH_WAYLAND_GAL2D
            elected = electSurfaceForPointerEvent(p.x, p.y,
                                            INPUT_DEVICE_POINTER);
#endif
            if (elected != NULL)
            {
                if (!transformGlobalToLocalCoordinates(elected, p.x, p.y))
                    elected = NULL;
            }
            break;

        case INPUT_STATE_AXIS:
            elected = _getPointerFocus();
            break;

        default:
            elected = NULL;
            LOG_WARNING("InputManager",
                        "Invalid input state, value=" << p.state);
        }
    }
    return elected;
}

Surface * InputManager::reportTouchEvent(PointVect& pv, int displayID)
{
    Surface * elected = NULL;
    PointVectIterator it;

    for (it = pv.begin(); it != pv.end(); it++)
    {
        switch (it->state)
        {
            case INPUT_STATE_PRESSED:
                if (INVALID_DISPLAY_ID == displayID)
                {
                    elected = electSurfaceForPointerEvent(it->x, it->y,
                                                    INPUT_DEVICE_TOUCH);

                }
                else
                {
                    elected = electSurfaceForPointerEventOnDisplay(it->x, it->y,
                                            INPUT_DEVICE_TOUCH, displayID);
                }
                _setTouchFocus(elected);
                break;

            case INPUT_STATE_OTHER:
            case INPUT_STATE_MOTION:
            case INPUT_STATE_RELEASED:
                elected = _getTouchFocus();
                if (elected != NULL)
                {
                    if (!transformGlobalToLocalCoordinates(elected, it->x, it->y))
                        elected = NULL;
                }
                break;

            default:
                elected = NULL;
                LOG_WARNING("InputManager",
                            "Invalid input state, value=" << it->state);
        }
    }
    return elected;
}

Surface * InputManager::reportTouchEvent(PointVect& pv)
{
    Surface * elected = NULL;
    PointVectIterator it;

    for (it = pv.begin(); it != pv.end(); it++)
    {
        switch (it->state)
        {
            case INPUT_STATE_PRESSED:
                elected = electSurfaceForPointerEvent(it->x, it->y, INPUT_DEVICE_TOUCH);
                _setTouchFocus(elected);
                break;

            case INPUT_STATE_OTHER:
            case INPUT_STATE_MOTION:
            case INPUT_STATE_RELEASED:
                elected = _getTouchFocus();
                if (elected != NULL)
                {
                    if (!transformGlobalToLocalCoordinates(elected, it->x, it->y))
                        elected = NULL;
                }
                break;

            default:
                elected = NULL;
                LOG_WARNING("InputManager",
                            "Invalid input state, value=" << it->state);
        }
    }
    return elected;
}

Surface * InputManager::electSurfaceFromDisplay(int& x, int& y, InputDevice input_dev, int displayID)
{
    LayerListConstReverseIterator currentLayer;
    SurfaceListConstReverseIterator currentSurf;
    int x_SurfCoordinate;
    int y_SurfCoordinate;
    LayerList &ll = m_pScene->getCurrentRenderOrder(displayID);
    Surface* surf = NULL;

    /* Need to browse for all layers. 1st layer of m_currentRenderOrder is rendered
     * on bottom, last one is rendrered on top. So we have to reverse iterate */
    LayerListConstReverseIterator layerEnd(ll.rend());
    for (currentLayer = ll.rbegin(); currentLayer != layerEnd && surf == NULL; currentLayer++)
    {
        if ((*currentLayer)->visibility
            && ((*currentLayer)->getOpacity() != 0)
            && (*currentLayer)->isInside(x, y))
        {
            x_SurfCoordinate = x;
            y_SurfCoordinate = y;
            (*currentLayer)->DestToSourceCoordinates(&x_SurfCoordinate, &y_SurfCoordinate, false);
            /* Need to browse for all surfaces */
            SurfaceListConstReverseIterator surfEnd((*currentLayer)->getAllSurfaces().rend());
            for (currentSurf = (*currentLayer)->getAllSurfaces().rbegin();
                 currentSurf !=  surfEnd && surf == NULL;
                 currentSurf++)
            {
                if ((*currentSurf)->hasNativeContent()
                    && (*currentSurf)->visibility
                    && ((*currentSurf)->getOpacity() != 0)
                    && (*currentSurf)->isInside(x_SurfCoordinate, y_SurfCoordinate)
                    && (*currentSurf)->isInputEventAcceptedFrom(input_dev)
                    // TODO: jge: This should ideally be covered by hasNativeContent()!
                    && (*currentSurf)->platform != NULL)
                {
                    (*currentSurf)->DestToSourceCoordinates(&x_SurfCoordinate, &y_SurfCoordinate, false);
                    /*check the transparency of the surface in the (x,y) coordinate
                     * this is only done if the graphical system is not set
                     * means feature is not set*/
                    if (m_pGraphicSystem == NULL)
                    {
                        surf = *currentSurf;
                    }
                    else
                    {
                        if (m_pGraphicSystem->isSurfaceTransparentOn(*currentSurf,
                                                                     x_SurfCoordinate,
                                                                     y_SurfCoordinate))
                        {
                            /*restore screen coordinates for next computations*/
                            x_SurfCoordinate = x;
                            y_SurfCoordinate = y;
                            continue;
                        }
                        else
                        {
                            surf = *currentSurf;
                        }
                    }
                    x = x_SurfCoordinate;
                    y = y_SurfCoordinate;
                }
            }
        }
    }
    return surf;
}

Surface * InputManager::electSurfaceForPointerEvent(int& x, int& y, InputDevice input_dev)
{
    Surface *surf = NULL;
    Surface *surf_ret = surf;
    m_pScene->lockScene();
    int current_disp_id;
    int dispid_for_input = m_config.getDisplayForInput();

    LmScreenList screenList = m_pScene->getScreenList();
    LmScreenListIterator iter = screenList.begin();
    LmScreenListIterator iterEnd = screenList.end();

    for (; iter != iterEnd; ++iter)
    {
        current_disp_id = (*iter)->getID();

        /*if specific display id for input is set evaluate only this
         * and skip all others displays*/
        if ((dispid_for_input != DEFAULT_DISP_ID_FOR_INPUT) &&
                (current_disp_id != dispid_for_input))
                continue;
        surf = electSurfaceFromDisplay(x, y, input_dev, current_disp_id);
        if (NULL != surf)
        {
            surf_ret = surf;
            /*Add this to exit when detect right surface to handle, not continue to handle it on other screen, 
	    because after handle on first screen the coordinate is change*/ 
            break;
        }
    }
    if (NULL != surf_ret)
    {
        if (false == surf_ret->checkSeatAcceptance(m_seatName))
        {
            std::string seatName(m_seatName);
            LOG_WARNING("InputManager", "Ignoring pointer/touch event because Surface:"
                   << surf_ret->getID() << " does not accept inputs from seat:" << seatName);
            surf_ret = NULL;
        }
    }
    m_pScene->unlockScene();
    return surf_ret;
}

Surface * InputManager::electSurfaceForPointerEventOnDisplay(int& x, int& y, InputDevice input_dev, int displayID)
{
    Surface *surf = NULL;
    m_pScene->lockScene();
    bool displayExists = false;
    int dispid_for_input = m_config.getDisplayForInput();

    LmScreenList screenList = m_pScene->getScreenList();
    LmScreenListIterator iter = screenList.begin();
    LmScreenListIterator iterEnd = screenList.end();

    surf = NULL;
    if ((dispid_for_input == DEFAULT_DISP_ID_FOR_INPUT) ||
            (displayID == dispid_for_input))
    {
        for (; iter != iterEnd; ++iter)
        {
            if (displayID == (int)(*iter)->getID())
            {
                displayExists = true;
            }
        }
        if (displayExists)
        {
            /*if specific display id for input is set evaluate only this
             * and skip all others displays*/
            surf = electSurfaceFromDisplay(x, y, input_dev, displayID);
            if (NULL != surf)
            {
                if (false == surf->checkSeatAcceptance(m_seatName))
                {
                    std::string seatName(m_seatName);
                    LOG_WARNING("InputManager", "Ignoring pointer/touch event because Surface:"
                           << surf->getID() << " does not accept inputs from seat:" << seatName);
                    surf = NULL;
                }
            }
        }
        else
        {
            LOG_ERROR("InputManager", "electSurfaceForPointerEventOnDisplay: could not find display id:" << displayID);
        }
    }
    m_pScene->unlockScene();
    return surf;
}

bool InputManager::transformGlobalToLocalCoordinates(Surface* surf, int& x, int& y)
{
    Layer* layer;

    assert(surf != NULL);

    layer = m_pScene->getLayer(surf->getContainingLayerId());
    if (layer != NULL)
    {
        layer->DestToSourceCoordinates(&x, &y, false);
        surf->DestToSourceCoordinates(&x, &y, false);
        return true;
    }
    else
    {/*work around for the application shutdown with event focus*/
        LOG_WARNING("InputManager",
                    "ERROR by surface setup, layer is missing "
                    "set input focus to NULL");
        if( _getPointerFocus() == surf)
            _setPointerFocus(NULL);
        if( _getTouchFocus() == surf)
            _setTouchFocus(NULL);
        if( _getKbdFocus() == surf)
            _setKbdFocus(NULL);
        return false;
    }
}


//@{
/*
 * m_KbdFocus can be concurrently accessed by :
 *   - reportKeyboardEvent(), called in the renderer thread context
 *   - setKeyboardFocusOn(), called in the IPC mechanism context
 *
 *   So read & Write to this variable must be protected by exclusive to avoid
 *   race conditions
 */

void InputManager::_setKbdFocus(Surface * s)
{
    pthread_mutex_lock(&m_mutex);
    if (NULL != m_KbdFocus)
    {
        m_KbdFocus->resetFocusBitmap(ILM_INPUT_DEVICE_KEYBOARD);
    }
    m_KbdFocus = s;
    if (NULL != s)
    {
        s->setFocusBitmap(ILM_INPUT_DEVICE_KEYBOARD);
    }
    pthread_mutex_unlock(&m_mutex);
}

Surface * InputManager::_getKbdFocus()
{
    Surface * s;
    pthread_mutex_lock(&m_mutex);
    s = m_KbdFocus;
    pthread_mutex_unlock(&m_mutex);

    return s;
}
//@}



//@{
/*
 * m_PointerFocus can be concurrently accessed by :
 *   - reportPointetEvent(), called in the renderer thread context
 *   - setPointerFocusOn(), called in the IPC mechanism context
 *
 *   So read & Write to this variable must be protected by exclusive to avoid
 *   race conditions
 */
void InputManager::_setPointerFocus(Surface * s)
{
    pthread_mutex_lock(&m_mutex);
    if (NULL != m_PointerFocus)
    {
        m_PointerFocus->resetFocusBitmap(ILM_INPUT_DEVICE_POINTER);
    }
    m_PointerFocus = s;
    if (NULL != s)
    {
        s->setFocusBitmap(ILM_INPUT_DEVICE_POINTER);
    }
    pthread_mutex_unlock(&m_mutex);
}

Surface * InputManager::_getPointerFocus()
{
    Surface * s;
    pthread_mutex_lock(&m_mutex);
    s = m_PointerFocus;
    pthread_mutex_unlock(&m_mutex);

    return s;
}
//@}


void InputManager::dropFocuses(unsigned int surfId)
{
    pthread_mutex_lock(&m_mutex);
    if (m_PointerFocus != NULL && m_PointerFocus->getID() == surfId)
    {
        m_PointerFocus = NULL;
    }

    if (m_KbdFocus != NULL && m_KbdFocus->getID() == surfId)
    {
        m_KbdFocus = NULL;
    }
    //Remove all references to destroyed surfaces from the map
    std::map<long, Surface*>::iterator it;
    Surface *surf;
    long key;
    for (it = m_KeyMap.begin(); it != m_KeyMap.end(); it++)
    {
        surf = it->second;
        key = it->first;
        if (NULL != surf)
        {
            if (surf->getID() == surfId)
            {
                LOG_DEBUG("InputManager",
                "keymap for key:" << key << " on surface ID:" << surfId << " set to NULL");
                m_KeyMap[key] = NULL;
            }
        }
    }

    if (m_TouchFocus != NULL && m_TouchFocus->getID() == surfId)
    {
        m_TouchFocus = NULL;
    }

    pthread_mutex_unlock(&m_mutex);
}

//@{
/*
 * m_TouchFocus can NOT be concurrently accessed as of today.
 * It is only accessed by reportPointerEvent() & reportTouchEvent(), both called in the renderer thread context.
 *
 * But it's cheap to add getter / setter if needed in the future & at least to have
 * a similar access mechanism than m_KbdFocus
 */
void InputManager::_setTouchFocus(Surface * s)
{
    pthread_mutex_lock(&m_mutex);
    if (NULL != m_TouchFocus)
    {
        m_TouchFocus->resetFocusBitmap(ILM_INPUT_DEVICE_TOUCH);
    }
    m_TouchFocus = s;
    if (NULL != s)
    {
        s->setFocusBitmap(ILM_INPUT_DEVICE_TOUCH);
    }
    pthread_mutex_unlock(&m_mutex);
}

Surface * InputManager::_getTouchFocus()
{
    Surface *s;
    pthread_mutex_lock(&m_mutex);
    s = m_TouchFocus;
    pthread_mutex_unlock(&m_mutex);

    return s;
}

void InputManager::setGraphicSystem(BaseGraphicSystem<void*, void*>* graphis_sys)
{
    m_pGraphicSystem = graphis_sys;
}

void InputManager::setSeatName(char* pSeatName)
{
    if (NULL != pSeatName)
    {
        strncpy(m_seatName, pSeatName, (MAX_SEAT_NAME_LNGTH - 2));
        m_seatName[MAX_SEAT_NAME_LNGTH - 1] = '\0';
    }
}

char* InputManager::getSeatName(void)
{
    return m_seatName;
}

bool InputManager::getCurrentDevices(ilmInputDevice *pBitmask)
{
    bool ret = false;
    if (NULL != pBitmask)
    {
        if (m_pKeyboardCount > 0)
        {
            *pBitmask |= ILM_INPUT_DEVICE_KEYBOARD;
        }
        if (m_pPointerCount > 0)
        {
            *pBitmask |= ILM_INPUT_DEVICE_POINTER;
        }
        if (m_pTouchCount > 0)
        {
            *pBitmask |= ILM_INPUT_DEVICE_TOUCH;
        }
        ret = true;
    }
    return ret;
}

void InputManager::removeDevices(ilmInputDevice deviceBitmask)
{
    if (deviceBitmask & ILM_INPUT_DEVICE_KEYBOARD)
    {
        m_pKeyboardCount--;
    }
    if (deviceBitmask & ILM_INPUT_DEVICE_POINTER)
    {
        m_pPointerCount--;
    }
    if (deviceBitmask & ILM_INPUT_DEVICE_TOUCH)
    {
        m_pTouchCount--;
    }
    LOG_DEBUG("InputManager","m_seatName = "<<m_seatName<<" this="<<this<<" m_pKeyboardCount="<<m_pKeyboardCount<<""
            "m_pPointerCount="<<m_pPointerCount<<" m_pTouchCount="<<m_pTouchCount);
}

void InputManager::addDevices(ilmInputDevice deviceBitmask)
{
    if (deviceBitmask & ILM_INPUT_DEVICE_KEYBOARD)
    {
        m_pKeyboardCount++;
    }
    if (deviceBitmask & ILM_INPUT_DEVICE_POINTER)
    {
        m_pPointerCount++;
    }
    if (deviceBitmask & ILM_INPUT_DEVICE_TOUCH)
    {
        m_pTouchCount++;
    }
    LOG_DEBUG("InputManager","m_seatName = "<<m_seatName<<" this="<<this<<" m_pKeyboardCount="<<m_pKeyboardCount<<""
            "m_pPointerCount="<<m_pPointerCount<<" m_pTouchCount="<<m_pTouchCount);
}
